#!/usr/bin/perl -w
#
#  $Id: alnDbFormat.pl,v 1.4 2008/04/04 20:59:04 whuang Exp $
#  fastHmm/fastBlast Alignment Tools
#  http://microbesonline.org/fasthmm (fasthmm@microbesonline.org)
#
#  Script for generating fastHmm-compatible alignment databases
#  Supports panther, gene3d, pfam, pirsf, smart, tigrfam
#  Requires source data and InterPro data in some cases (see usage)
#
#  Copyright (C) 2007 The Regents of the University of California
#  All rights reserved.
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
#  Disclaimer
#
#  NEITHER THE UNITED STATES NOR THE UNITED STATES DEPARTMENT OF ENERGY,
#  NOR ANY OF THEIR EMPLOYEES, MAKES ANY WARRANTY, EXPRESS OR IMPLIED,
#  OR ASSUMES ANY LEGAL LIABILITY OR RESPONSIBILITY FOR THE ACCURACY,
#  COMPLETENESS, OR USEFULNESS OF ANY INFORMATION, APPARATUS, PRODUCT,
#  OR PROCESS DISCLOSED, OR REPRESENTS THAT ITS USE WOULD NOT INFRINGE
#  PRIVATELY OWNED RIGHTS.
#

use strict;
use lib "$ENV{FASTHMM_DIR}/lib";
use Args;
use File;

#
# USAGE
#
my $usage =
"Usage:
  alnDbFormat.pl <options>

Parameters:
  -g <tar.gz>	Specify gene3d alignments tar.gz file to build

  -n <tar.gz>	Specify panther alignments+hmm tar.gz file to build

  -r <tar.gz>	Specify pirsf alignments tar.gz file to build

  -p <pfamBase>	Specify Pfam mirror base directory; must contain
		   Pfam-A.seed[.gz], Pfam-C[.gz], Pfam_fs[.gz], Pfam_ls[.gz]

  -s <tar.gz>	Specify smart alignments+hmm tar.gz file to build

  -ua <tar.gz>	Specify superfam alignment tar.gz file to build
  -uh <tar.gz>	Specify superfam hmm tar.gz file to build

  -ta <tar.gz>	Specify tigrfam alignment tar.gz file to build
  -th <tar.gz>	Specify tigrfam hmm tar.gz file to build

Required for -g, -r, -ua/-uh:
  -i <tar.gz>	Specify InterProScan data file for hmms

Optional Parameters:
  -c <thresh>	cdhit reduction for alignments with over <thresh> seqs
		   (except superfam)
  -ck <inFa>	Build psiblast checkpoint files using inFa fasta seqs
		   (except for superfam)
  -ckj <procs>	Build psiblast checkpoint files using <procs> processes
		   (Default: 1)
  -h		Build \"hard\" HMM lists (except gene3d, superfam)
  -o <outDir>	Directory in which to create reformatted database directories
		   (Default: Use FASTHMM_DIR/db or ./ if former doesn't exist)
  -t <tmpDir>	Temporary directory (Default: use /tmp)
  -q		Quiet execution; disable status updates to stdout

";

my %typeFileFilter = (
			'panther'	=> [ '^cluster\.pir$' ],
			'gene3d'	=> [ '\.alignment$' ],
			'pfam'		=> [ 'Pfam-A\.seed(?:\.gz)?$' ],
			'pirsf'		=> [ '\.seed$' ],
			'smart'		=> [ '\.aln$' ],
			'tigrfam'	=> [ '\.SEED$' ],
		     );

#
# Parse Command Line Options
#
my ($opts, $nonOpts) = Args::getArgs( 
	"g:;i|n:|r:;i|p:|s:|+;uh;ua:;uh*i|+;ua;uh:;ua*i|+;th;ta:;th|+;ta;th:;ta|+;g+r+(ua*uh);i:|o:|q|c:|ck:|ckj:|h|t:",
	@ARGV, -1, $usage );

if ( !exists( $opts->{o} ) )
{
	my $dbDir = $ENV{FASTHMM_DIR} . "/db";
	if ( -d $dbDir )
	{
		$opts->{o} = $dbDir;
	} else {
		$opts->{o} = ".";
	}
}

$ENV{FASTHMM_DIR} = "."
	if ( !exists( $ENV{FASTHMM_DIR} ) );

$opts->{ckj} = 1
	if ( !exists( $opts->{ckj} ) );
$opts->{t} = "/tmp"
	if ( !exists( $opts->{t} ) );

if ( !(-d $opts->{o}) )
{
	mkdir( $opts->{o}, 0755 );
	die "Error: Specified output directory '", $opts->{o}, "' does not exist!\n"
		if ( !(-d $opts->{o}) );
}

die $usage
	if ( !exists( $opts->{g} ) &&
		!exists( $opts->{n} ) &&
		!exists( $opts->{r} ) &&
		!exists( $opts->{p} ) &&
		!exists( $opts->{s} ) &&
		!exists( $opts->{ua} ) &&
		!exists( $opts->{ta} ) );

alnDbFormat( $opts );

exit(0);


sub extractInterProHmms
{
	my $opts = shift;
	my $iprDataFile = $opts->{i};
	my $hmmFiles = shift;

	my $iprDataDir;

	if ( $iprDataFile =~ /(?:\.tar\.gz|\.tgz)$/i )
	{
		$iprDataDir = $opts->{t} . "/iprData_$$";
		mkdir( $iprDataDir, 0755 );

		my $cmd = $ENV{FASTHMM_DIR} . "/bin/tar zxvf $iprDataFile -C $iprDataDir " . join(" ", @{$hmmFiles});
		print "Extracting required InterProScan HMMs ...\n"
			if ( !exists( $opts->{q} ) );

		#print "cmd:[$cmd]\n"
		#	if ( !exists( $opts->{q} ) );
		system( $cmd . " >/dev/null 2>&1" );
	} elsif ( -d $iprDataFile )
	{
		$iprDataDir = $iprDataFile;
	} else
	{
		die "Error: You must specify a valid tar.gz archive or a directory!";
	}

	return $iprDataDir;
}

sub processAlignmentData
{
	my $opts = shift;
	my $inDir = shift;
	my $filters = shift;
	my $type = shift;
	print "Building filtered file list ... \n"
		if ( !exists( $opts->{q} ) );
	my $files = File::getFilteredFileList( $inDir, $filters );
	my $numFiles = scalar( @{$files} );
	print "Found $numFiles file(s) matching pre-defined filters ...\n"
		if ( !exists( $opts->{q} ) );

	if ( $numFiles > 0 )
	{
		reformatDb( $opts, $type, $files, $numFiles );
	} else {
		print STDERR "Warning: Nothing to do - possibly a corrupt source archive for $type\n";
	}

	return ( $files, $numFiles );
}

sub alnDbFormat_gene3d
{
	my $opts = shift;
	my $iprDataDir = shift;
	my $hmmFile = $iprDataDir . "/iprscan/data/Gene3D.hmm";

	my $outDir = $opts->{o} . "/gene3d";
	mkdir( $outDir, 0755 )
		if ( !(-d $outDir) );

	# extract data
	my $extract = 0;
	my $tmpDir;
	if ( $opts->{g} =~ /(?:\.tar\.gz|\.tgz)$/i )
	{
		$extract = 1;
		$tmpDir = $opts->{t} . "/alnDbFormat_$$";
		mkdir( $tmpDir, 0755 )
			if ( !(-d $tmpDir) );

		my $cmd = $ENV{FASTHMM_DIR} . "/bin/tar zxvf $opts->{g} -C $tmpDir";
		print "Extracting gene3d alignments ...\n"
			if ( !exists( $opts->{q} ) );
		#print "cmd:[$cmd]\n"
		#	if ( !exists( $opts->{q} ) );
		system( $cmd . " >/dev/null 2>&1" );
	} elsif ( -d $opts->{g} )
	{
		$tmpDir = $opts->{g};
	} else
	{
		die "Error: You must specify a valid tar.gz archive or a directory!";
	}

	my $filters = $typeFileFilter{'gene3d'};
	processAlignmentData( $opts, $tmpDir, $filters, 'gene3d' );

	# extract hmm data from iprscan
	print "Extracting gene3d HMMs ...\n"
		if ( !exists( $opts->{q} ) );
	my $outHmmDir = $outDir . "/hmm";
	mkdir( $outHmmDir, 0755 )
		if ( !(-d $outHmmDir) );

	local *IN;
	local *OUT;
	local *SF;
	my $nModels = 0;
	my $sfTabFile = $outDir . "/gene3d.tab";
	open( SF, ">$sfTabFile" );
	open( IN, "<$hmmFile" );
	my $buffer = "";
	my $sfName = undef;
	while(<IN>)
	{
		$buffer .= $_;
		if ( /^ACC\s+(\S+)/ )
		{
			$sfName = $1;
		} elsif ( /\-\-histfile\s+([^.]+)/i )
		{
			my $acc = $1;
			my $outHmmFile = $outHmmDir . "/$acc.hmm";
			print SF join("\t", $acc, $sfName), "\n";
			open(OUT, ">$outHmmFile");
			my $line;
			print OUT $buffer;
			do
			{
				$line = <IN>;
				print OUT $line;
			} while ( $line !~ /^\/\// );
			close(OUT);

			$nModels++;
			$buffer = "";
		}
	}
	close(IN);
	close(SF);

	if ( $extract )
	{
		print "Cleaning up ...\n"
			if ( !exists( $opts->{q} ) );
		my $cmd = $ENV{FASTHMM_DIR} . "/bin/rm -rf $tmpDir";
		system( $cmd . " >/dev/null 2>&1" );
	}

	return $nModels;
}

sub alnDbFormat_panther
{
	my $opts = shift;

	my $outDir = $opts->{o} . "/panther";
	mkdir( $outDir, 0755 )
		if ( !(-d $outDir) );

	my $extract = 0;
	my $tmpDir;
	if ( $opts->{n} =~ /(?:\.tar\.gz|\.tgz)$/i )
	{
		# extract data
		$extract = 1;
		$tmpDir = $opts->{t} . "/alnDbFormat_$$";
		mkdir( $tmpDir, 0755 )
		if ( !(-d $tmpDir) );

		my $cmd = $ENV{FASTHMM_DIR} . "/bin/tar zxvf $opts->{n} -C $tmpDir";
		print "Extracting panther alignments and HMMs ...\n"
			if ( !exists( $opts->{q} ) );
		#print "cmd:[$cmd]\n"
		#	if ( !exists( $opts->{q} ) );
		system( $cmd . " >/dev/null 2>&1" );
	} elsif ( -d $opts->{n} )
	{
		$tmpDir = $opts->{n};
	} else
	{
		die "Error: You must specify a valid tar.gz archive or a directory!";
	}

	my $filters = $typeFileFilter{'panther'};
	my ( $files, $numFiles ) = processAlignmentData( $opts, $tmpDir, $filters, 'panther' );

	# extract hmm data
	print "Extracting panther HMMs ...\n"
		if ( !exists( $opts->{q} ) );
	my $outHmmDir = $outDir . "/hmm";
	mkdir( $outHmmDir, 0755 )
		if ( !(-d $outHmmDir) );

	local *IN;
	local *OUT;

	my $nModels = scalar( @{$files} );
	foreach my $file ( @{$files} )
	{
		my @parts = split(/\//, $file);
		$parts[-1] = 'hmmer.hmm';
		my $inFile = join("/", @parts);
		my $outFile = $outHmmDir . "/" . $parts[-2] . ".hmm";

		open(IN, "<$inFile");
		open(OUT, ">$outFile");
		while(<IN>)
		{
			print OUT;
		}
		close(OUT);
		close(IN);
	}

	# names.tab
	print "Copying panther names.tab ...\n"
		if ( !exists( $opts->{q} ) );
	my $pantherBaseDir = $files->[0];
	$pantherBaseDir =~ s/\/books\/\w+\/\S+//ig;
	my $namesFile = $pantherBaseDir . "/globals/names.tab";
	my $outNamesFile = $outDir . "/panther.names";
	open(IN, "<$namesFile");
	open(OUT, ">$outNamesFile");
	while(<IN>)
	{
		print OUT;
	}
	close(OUT);
	close(IN);

	# prepare books hierarchy
	print "Preparing panther books hierarchy ...\n"
		if ( !exists( $opts->{q} ) );
	my $booksDir = $outDir . "/books";
	mkdir( $booksDir, 0755 )
		if ( !(-d $booksDir) );
	foreach my $file ( @{$files} )
	{
		my @parts = split(/\//, $file);
		my $acc = $parts[-2];
		my $accDir = $file;
		$accDir =~ s/\/cluster\.pir$//ig;
		my $hmmFiles = File::getFilteredFileList( $accDir, [ 'hmmer\.hmm$' ] );
		my $outAccDir = $booksDir . "/$acc";
		mkdir( $outAccDir, 0755 )
			if ( !(-d $outAccDir) );
		foreach my $hmmFile ( @{$hmmFiles} )
		{
			my @parts = split(/\//, $hmmFile);
			if ( $parts[-2] ne $acc )
			{
				my $sfName = $parts[-2];
				my $sfDir = $outAccDir . "/$sfName";
				mkdir( $sfDir, 0755 )
					if ( !(-d $sfDir) );
				my $outFile = $sfDir . "/hmmer.hmm";

				open(IN, "<$hmmFile");
				open(OUT, ">$outFile");
				while(<IN>)
				{
					print OUT;
				}
				close(OUT);
				close(IN);
			}
		}
	}

	if ( $extract )
	{
		print "Cleaning up ...\n"
			if ( !exists( $opts->{q} ) );
		my $cmd = $ENV{FASTHMM_DIR} . "/bin/rm -rf $tmpDir";
		system( $cmd . " >/dev/null 2>&1" );
	}

	return $nModels;
}

sub alnDbFormat_pirsf
{
	my $opts = shift;
	my $iprDataDir = shift;
	my $hmmFile = $iprDataDir . "/iprscan/data/sf_hmm";
	my $pirsfDatFile = $iprDataDir . "/iprscan/data/pirsf.dat";

	my $outDir = $opts->{o} . "/pirsf";
	mkdir( $outDir, 0755 )
		if ( !(-d $outDir) );

	# extract data
	my $extract = 0;
	my $tmpDir;
	if ( $opts->{r} =~ /(?:\.tar\.gz|\.tgz)$/i )
	{
		$extract = 1;
		$tmpDir = $opts->{t} . "/alnDbFormat_$$";
		mkdir( $tmpDir, 0755 )
			if ( !(-d $tmpDir) );

		my $cmd = $ENV{FASTHMM_DIR} . "/bin/tar zxvf $opts->{r} -C $tmpDir";
		print "Extracting pirsf alignments ...\n"
			if ( !exists( $opts->{q} ) );
		#print "cmd:[$cmd]\n"
		#	if ( !exists( $opts->{q} ) );
		system( $cmd . " >/dev/null 2>&1" );
	} elsif ( -d $opts->{r} )
	{
		$tmpDir = $opts->{r};
	} else
	{
		die "Error: You must specify a valid tar.gz archive or a directory!";
	}

	my $filters = $typeFileFilter{'pirsf'};
	processAlignmentData( $opts, $tmpDir, $filters, 'pirsf' );

	# extract hmm data from iprscan
	print "Extracting pirsf HMMs ...\n"
		if ( !exists( $opts->{q} ) );
	my $outHmmDir = $outDir . "/hmm";
	mkdir( $outHmmDir, 0755 )
		if ( !(-d $outHmmDir) );

	local *IN;
	local *OUT;
	my $nModels = 0;
	open( IN, "<$hmmFile" );
	my $buffer = "";
	while(<IN>)
	{
		$buffer .= $_;
		if ( /^ACC\s+(\S+)/i )
		{
			my $acc = $1;
			my $outHmmFile = $outHmmDir . "/$acc.hmm";
			open(OUT, ">$outHmmFile");
			my $line;
			print OUT $buffer;
			do
			{
				$line = <IN>;
				print OUT $line;
			} while ( $line !~ /^\/\// );
			close(OUT);

			$nModels++;
			$buffer = "";
		}
	}
	close(IN);
	open( IN, "<${hmmFile}_sub" );
	$buffer = "";
	while(<IN>)
	{
		$buffer .= $_;
		if ( /^ACC\s+(\S+)/i )
		{
			my $acc = $1;
			my $outHmmFile = $outHmmDir . "/$acc.hmm";
			open(OUT, ">$outHmmFile");
			my $line;
			print OUT $buffer;
			do
			{
				$line = <IN>;
				print OUT $line;
			} while ( $line !~ /^\/\// );
			close(OUT);

			$nModels++;
			$buffer = "";
		}
	}
	close(IN);

	print "Copying pirsf.dat ...\n"
		if ( !exists( $opts->{q} ) );
	open(IN, "<$pirsfDatFile");
	open(OUT, ">$outDir/pirsf.dat");
	while(<IN>)
	{
		print OUT;
	}
	close(IN);
	close(OUT);

	if ( $extract )
	{
		print "Cleaning up ...\n"
			if ( !exists( $opts->{q} ) );
		my $cmd = $ENV{FASTHMM_DIR} . "/bin/rm -rf $tmpDir";
		system( $cmd . " >/dev/null 2>&1" );
	}

	return $nModels;
}

sub alnDbFormat_pfam
{
	my $opts = shift;

	my $outDir = $opts->{o} . "/pfam";
	mkdir( $outDir, 0755 )
		if ( !(-d $outDir) );

	# check for required data
	my $seedFile = File::fileExtensionPri( $opts->{p} . "/Pfam-A.seed", "gz" );
	my $clanFile = File::fileExtensionPri( $opts->{p} . "/Pfam-C", "gz" );
	my $lsHmmFile = File::fileExtensionPri( $opts->{p} . "/Pfam_ls", "gz" );
	my $fsHmmFile = File::fileExtensionPri( $opts->{p} . "/Pfam_fs", "gz" );
	if ( !defined($seedFile) ||
		!defined($clanFile) ||
		!defined($lsHmmFile) ||
		!defined($fsHmmFile) )
	{
		print STDERR "Error: One or more required Pfam files are missing; skipping Pfam build ...\n";
		return 0;
	}

	my $filters = $typeFileFilter{'pfam'};
	processAlignmentData( $opts, $opts->{p}, $filters, 'pfam' );

	# build pfam to clan mapping
	print "Building pfam to clan mapping ...\n"
		if ( !exists( $opts->{q} ) );
	# the clan file does not have the full pfam member accessions. we need to build a map
	# and other bs. lame.

	# get a list of .b files we created
	my $files = File::getFilteredFileList( $outDir, [ '\.b$' ] );
	my %pfAccMap = ();
	foreach my $file ( @{$files} )
	{
		my @parts = split(/\//, $file);
		my $baseFile = $parts[-1];
		$baseFile =~ s/\.b$//i;
		if ( $baseFile =~ /^(\w+)\.\d+$/ )
		{
			$pfAccMap{$1} = $baseFile;
		} else {
			print STDERR "Warning: Pfam accession $baseFile does not match expected pattern\n";
		}
	}

	my $clanMapFile = $outDir . "/Pfam-C.map";
	local *IN;
	local *OUT;

	File::fileGzOpen( \*IN, $clanFile );
	open(OUT, ">$clanMapFile");
	my $curClanAcc = undef;
	while(<IN>)
	{
		if ( /^AC\s+(\S+)/ )
		{
			$curClanAcc = $1;
		} elsif ( defined($curClanAcc) && /^MB\s+(\w+)/ )
		{
			print OUT join("\t", $curClanAcc, $pfAccMap{$1}), "\n";
		}
	}
	close(OUT);
	close(IN);


	print "Extracting pfam HMMs ...\n"
		if ( !exists( $opts->{q} ) );
	my $outHmmDir = $outDir . "/hmm";
	mkdir( $outHmmDir, 0755 )
		if ( !(-d $outHmmDir) );

	local *IN;
	local *OUT;
	my $nModels = 0;
	File::fileGzOpen( \*IN, $lsHmmFile );
	my $buffer = "";
	while(<IN>)
	{
		$buffer .= $_;
		if ( /^ACC\s+(\w+\.\d+)/i )
		{
			my $acc = $1;
			my $outHmmFile = $outHmmDir . "/$acc.hmm";
			open(OUT, ">$outHmmFile");
			my $line;
			print OUT $buffer;
			do
			{
				$line = <IN>;
				print OUT $line;
			} while ( $line !~ /^\/\// );
			close(OUT);

			$nModels++;
			$buffer = "";
		}
	}
	close(IN);


	print "Extracting pfam fs HMMs ...\n"
		if ( !exists( $opts->{q} ) );
	File::fileGzOpen( \*IN, $fsHmmFile );
	$buffer = "";
	my @fsModels = ();
	while(<IN>)
	{
		$buffer .= $_;
		if ( /^ACC\s+(\w+\..+)/i )
		{
			my $acc = $1;
			$acc .= ".fs"
				if ( $acc !~ /\.fs$/i );

			my $outHmmFile = $outHmmDir . "/$acc.hmm";
			open(OUT, ">$outHmmFile");
			my $line;
			print OUT $buffer;
			do
			{
				$line = <IN>;
				print OUT $line;
			} while ( $line !~ /^\/\// );
			close(OUT);
			push( @fsModels, $acc );

			$nModels++;
			$buffer = "";
		}
	}
	close(IN);

	# convert hmm's to psiblast/seq files
	print "Converting pfam fs HMMs ...\n"
		if ( !exists( $opts->{q} ) );
	foreach my $acc ( @fsModels )
	{
		my $hmmFile = $outHmmDir . "/$acc.hmm";
		my $alnFilePsi = $outDir . "/$acc.psiblast";
		my $seqFile = $outDir . "/$acc.seq";
		
		my $cmd = $ENV{FASTHMM_DIR} . "/bin/hmmToPsiBlast.pl -in $hmmFile -out $alnFilePsi -seed $seqFile";
		system( $cmd . " >/dev/null 2>&1" );
		print "Warning: Could not convert pfam fs model '$acc'; skipping ...\n"
			if ( !(-e $alnFilePsi) || !(-e $seqFile) );
	}

	return $nModels;
}

sub alnDbFormat_smart
{
	my $opts = shift;

	my $outDir = $opts->{o} . "/smart";
	mkdir( $outDir, 0755 )
		if ( !(-d $outDir) );

	my $extract = 0;
	my $tmpDir;
	if ( $opts->{s} =~ /(?:\.tar\.gz|\.tgz)$/i )
	{
		# extract data
		$extract = 1;
		$tmpDir = $opts->{t} . "/alnDbFormat_$$";
		mkdir( $tmpDir, 0755 )
		if ( !(-d $tmpDir) );

		my $cmd = $ENV{FASTHMM_DIR} . "/bin/tar zxvf $opts->{s} -C $tmpDir";
		print "Extracting smart alignments and HMMs ...\n"
			if ( !exists( $opts->{q} ) );
		#print "cmd:[$cmd]\n"
		#	if ( !exists( $opts->{q} ) );
		system( $cmd . " >/dev/null 2>&1" );
	} elsif ( -d $opts->{s} )
	{
		$tmpDir = $opts->{s};
	} else
	{
		die "Error: You must specify a valid tar.gz archive or a directory!";
	}

	my $filters = $typeFileFilter{'smart'};
	my ( $files, $numFiles ) = processAlignmentData( $opts, $tmpDir, $filters, 'smart' );

	# extract hmm data
	print "Extracting smart HMMs ...\n"
		if ( !exists( $opts->{q} ) );
	my $outHmmDir = $outDir . "/hmm";
	mkdir( $outHmmDir, 0755 )
		if ( !(-d $outHmmDir) );

	local *IN;
	local *OUT;

	my $nModels = scalar( @{$files} );

	foreach my $file ( @{$files} )
	{
		$file =~ s/\/aln\//\/hmm\//;
		$file =~ s/\.aln$//;
		my @parts = split(/\//, $file);
		my $inFile = $file . ".HMM";
		my $outFile = $outHmmDir . "/" . $parts[-1] . ".hmm";

		open(IN, "<$inFile");
		open(OUT, ">$outFile");
		while(<IN>)
		{
			print OUT;
		}
		close(OUT);
		close(IN);
	}

	if ( $extract )
	{
		print "Cleaning up ...\n"
			if ( !exists( $opts->{q} ) );
		my $cmd = $ENV{FASTHMM_DIR} . "/bin/rm -rf $tmpDir";
		system( $cmd . " >/dev/null 2>&1" );
	}

	return $nModels;
}

sub alnDbFormat_superfam
{
	my $opts = shift;
	my $iprDataDir = shift;

	my $sfTabFile = $iprDataDir . "/iprscan/data/superfamily.tab";

	my $outDir = $opts->{o} . "/superfam";

	if ( $opts->{ua} =~ /(?:\.tar\.gz|\.tgz)$/i )
	{
		my $cmd = $ENV{FASTHMM_DIR} . "/bin/tar zxvf $opts->{ua} -C $opts->{o}";
		print "Extracting superfam alignments ...\n"
			if ( !exists( $opts->{q} ) );
		#print "cmd:[$cmd]\n"
		#	if ( !exists( $opts->{q} ) );
		system( $cmd . " >/dev/null 2>&1" );
	} else
	{
		die "Error: You must specify a valid superfam alignment tar.gz archive!";
	}

	rename( $opts->{o} . "/psimodlib", $outDir );

	if ( $opts->{uh} =~ /(?:\.tar\.gz|\.tgz)$/i )
	{
		my $cmd = $ENV{FASTHMM_DIR} . "/bin/tar zxvf $opts->{uh} -C $outDir";
		print "Extracting superfam HMMs ...\n"
			if ( !exists( $opts->{q} ) );
		#print "cmd:[$cmd]\n"
		#	if ( !exists( $opts->{q} ) );
		system( $cmd . " >/dev/null 2>&1" );
	} else
	{
		die "Error: You must specify a valid superfam HMM tar.gz archive!";
	}

	rename( $outDir . "/hmmermodlib", $outDir . "/hmm" );

	# duplicate superfamily.tab file
	my $outSfTabFile = $opts->{o} . "/superfam/superfam.tab";
	local *IN;
	local *OUT;
	open(IN, "<$sfTabFile");
	open(OUT, ">$outSfTabFile");
	while(<IN>)
	{
		my @data = split(/\t/, $_);
		$data[1] = "SSF" . $data[1];
		print OUT join("\t", @data);
	}
	close(OUT);
	close(IN);

	return -1;
}

sub alnDbFormat_tigrfam
{
	my $opts = shift;

	my $outDir = $opts->{o} . "/tigrfam";
	mkdir( $outDir, 0755 )
		if ( !(-d $outDir) );

	my $extract = 0;
	my $tmpDir;
	if ( $opts->{ta} =~ /(?:\.tar\.gz|\.tgz)$/i )
	{
		# extract data
		$extract = 1;
		$tmpDir = $opts->{t} . "/alnDbFormat_$$";
		mkdir( $tmpDir, 0755 )
		if ( !(-d $tmpDir) );

		my $cmd = $ENV{FASTHMM_DIR} . "/bin/tar zxvf $opts->{ta} -C $tmpDir";
		print "Extracting tigrfam alignments and HMMs ...\n"
			if ( !exists( $opts->{q} ) );
		#print "cmd:[$cmd]\n"
		#	if ( !exists( $opts->{q} ) );
		system( $cmd . " >/dev/null 2>&1" );

		$cmd = $ENV{FASTHMM_DIR} . "/bin/tar zxvf $opts->{th} -C $tmpDir";
		system( $cmd . " >/dev/null 2>&1" );
	} elsif ( -d $opts->{ta} )
	{
		$tmpDir = $opts->{ta};
	} else
	{
		die "Error: You must specify a valid tar.gz archive or a directory!";
	}

	my $filters = $typeFileFilter{'tigrfam'};
	processAlignmentData( $opts, $tmpDir, $filters, 'tigrfam' );

	# extract hmm data
	print "Moving tigrfam HMMs ...\n"
		if ( !exists( $opts->{q} ) );
	my $outHmmDir = $outDir . "/hmm";
	mkdir( $outHmmDir, 0755 )
		if ( !(-d $outHmmDir) );

	local *IN;
	local *OUT;

	my $files = File::getFilteredFileList( $tmpDir, [ '\.HMM$' ] );

	my $nModels = scalar( @{$files} );
	foreach my $file ( @{$files} )
	{
		my $inFile = $file;
		my @parts = split(/\//, $file);
		$parts[-1] =~ s/\.HMM$/\.hmm/;
		my $outFile = $outHmmDir . "/" . $parts[-1];

		open(IN, "<$inFile");
		open(OUT, ">$outFile");
		while(<IN>)
		{
			print OUT;
		}
		close(OUT);
		close(IN);
	}

	if ( $extract )
	{
		print "Cleaning up ...\n"
			if ( !exists( $opts->{q} ) );
		my $cmd = $ENV{FASTHMM_DIR} . "/bin/rm -rf $tmpDir";
		system( $cmd . " >/dev/null 2>&1" );
	}

	return $nModels;
}

sub alnDbFormat
{
	my $opts = shift;

	my @iprFiles = ();
	push( @iprFiles, "*/Gene3D.hmm" )
		if ( exists( $opts->{g} ) );
	push( @iprFiles, "*/sf_hmm", "*/sf_hmm_sub", "*/pirsf.dat" )
		if ( exists( $opts->{r} ) );
	push( @iprFiles, "*/superfamily.tab" )
		if ( exists( $opts->{ua} ) );
	my $iprDataDir = extractInterProHmms( $opts, \@iprFiles )
		if ( exists($opts->{i}) && (scalar(@iprFiles) > 0) );

	my %nModels = ();

	$nModels{gene3d} = alnDbFormat_gene3d( $opts, $iprDataDir )
		if ( exists( $opts->{g} ) );

	$nModels{panther} = alnDbFormat_panther( $opts )
		if ( exists( $opts->{n} ) );

	$nModels{pirsf} = alnDbFormat_pirsf( $opts, $iprDataDir )
		if ( exists( $opts->{r} ) );

	$nModels{pfam} = alnDbFormat_pfam( $opts )
		if ( exists( $opts->{p} ) );

	$nModels{smart} = alnDbFormat_smart( $opts )
		if ( exists( $opts->{s} ) );

	$nModels{superfam} = alnDbFormat_superfam( $opts, $iprDataDir )
		if ( exists( $opts->{ua} ) );

	$nModels{tigrfam} = alnDbFormat_tigrfam( $opts )
		if ( exists( $opts->{ta} ) );

	# cdhit?
	my @reduceDbs = ();
	push( @reduceDbs, "gene3d" )	if ( exists( $opts->{g} ) );
	push( @reduceDbs, "panther" )	if ( exists( $opts->{n} ) );
	push( @reduceDbs, "pirsf" )	if ( exists( $opts->{r} ) );
	push( @reduceDbs, "pfam" )	if ( exists( $opts->{p} ) );
	push( @reduceDbs, "smart" )	if ( exists( $opts->{s} ) );
	push( @reduceDbs, "tigrfam" )	if ( exists( $opts->{ta} ) );

	if ( exists( $opts->{c} ) )
	{
		foreach my $db ( @reduceDbs )
		{
			print "cdhit-reducing $db ...\n"
				if ( !exists( $opts->{q} ) );
			my $files = File::getFilteredFileList( $opts->{o} . "/" . $db, [ '\.b$' ] );
			my $slots = $opts->{ckj};
			my $numFiles = scalar( @{$files} );

			for ( my $t = 0; $t < $numFiles; $t++ )
			{
				my $file = $files->[$t];

				if ( $slots > 0 )
				{
					my $childPid = fork();
					if ( defined($childPid) )
					{
						if ( $childPid == 0 )
						{
							$file =~ s/\.b$//g;
							my $cmd = $ENV{FASTHMM_DIR} . "/bin/cdhitReduce.pl $file $opts->{c}";
							system( $cmd . " >/dev/null 2>&1" );
							exit;
						}
					} else {
						die "unable to create child process to run cdhit\n";
					}

					$slots--;
				} else {
					$t--;
					my $childPid = wait();
					if ( $childPid > 0 )
					{
						$slots++;
					} elsif ( $childPid < 0 )
					{
						$slots = $opts->{ckj};
					}
				}
			}
		}

		# wait for jobs to finish before moving to next step
		my $childPid = 0;
		do
		{
			$childPid = wait();
		} until ( $childPid < 0 );
	}

	if ( exists( $opts->{ck} ) || exists( $opts->{h} ) )
	{
		# checkpointing using fastHmm.pl
		my $inFa = $opts->{ck};
		my $j = $opts->{ckj};

		foreach my $db ( @reduceDbs )
		{
			my $cmd = $ENV{FASTHMM_DIR} . "/bin/fastHmm.pl -i $inFa -t $db -C -f -r -d $opts->{o} -o $opts->{t} -j $j";
			print "building psiblast checkpoint files for $db ...\n"
				if ( !exists( $opts->{q} ) );
			system( $cmd . " >/dev/null 2>&1" );
			my @parts = split(/\//, $inFa);
			my ( $faBase ) = $parts[-1] =~ /^(\w+)/;
			unlink( $opts->{t} . "/result.$faBase.$db.domains" );
		}
	}

	if ( exists( $opts->{h} ) )
	{
		# build hard hmm list
		foreach my $db ( @reduceDbs )
		{
			next if ( ($db eq 'gene3d') || ($db eq 'superfam') );

			my $cmd;
			if ( $db =~ /fam$/i )
			{
				$cmd = $ENV{FASTHMM_DIR} . "/bin/listBadHMMs.pl -db $db -datadir $opts->{o} -nmodels 0 -numthreads $opts->{ckj}";
			} else {
				$cmd = $ENV{FASTHMM_DIR} . "/bin/listBadHMMs.pl -db $db -datadir $opts->{o} -nmodels $nModels{$db} -numthreads $opts->{ckj}";
			}

			print "building hard HMM list for $db ...\n"
				if ( !exists( $opts->{q} ) );
			system( $cmd . " >/dev/null 2>&1" );
		}
	}
}

# This function adapted from Fasta2B.pl selects from a list of
# sequences the one with the least amount of gap columns from
# all columns having >= 60% non-gap characters
sub getMorganBest
{
	my $seqs = shift;
	my $numSeqs = shift;
	my $seqLen = length( $seqs->[0]->{seq} );

	$numSeqs = scalar( @{$seqs} )
		if ( !defined($numSeqs) || ( $numSeqs < 1 ) );
	return undef
		if ( $numSeqs < 1 );

	my %scores = ();
	for ( my $i = 0; $i < $seqLen; $i++ )
	{
		my $col = "";
		for ( my $j = 0; $j < $numSeqs; $j++ )
		{
			$col .= substr( $seqs->[$j]->{seq}, $i, 1 );
		}

		my $gaps = $col =~ tr/-xX/-xX/;
		if ( $gaps < 0.6*$numSeqs )
		{
			for ( my $j = 0; $j < $numSeqs; $j++ )
			{
				$scores{$j}++
					if ( substr( $col, $j, 1 ) !~ /^[-xX]$/ );
			}
		}
	}

	my $bestIdx = 0;
	my $bestScore = 0;
	foreach my $i ( sort { $a <=> $b } keys( %scores ) )
	{
		if ( $scores{$i} > $bestScore )
		{
			$bestIdx = $i;
			$bestScore = $scores{$i};
		}
	}

	return $seqs->[$bestIdx];
}

sub alnFileIter
{
	my $fp = shift;
	my $type = shift;

	# read first non-empty line from the file pointer and try to
	# determine the alignment file format
	my $line = <$fp>;
	while ( defined( $line ) && ( $line =~ /^\s*$/ ) )
	{
		$line = <$fp>;
	}

	return undef
		if ( !defined( $line ) );

	if ( $line =~ /^>Subfamily:/i )
	{
		return fileIterFasta( $fp, $line, "^Subfamily:", ",\\s+\\d+\\s+bases,\\s+\\w+\\s+checksum\\.", "" );
	} elsif ( $line =~ /^>/ )
	{
		if ( defined($type) && ($type eq 'gene3d') )
		{
			return fileIterFasta( $fp, $line, undef, "\\s+No definition line found\\s*", "", $type );
		} elsif ( defined($type) && ($type eq 'panther') )
		{
			return fileIterFasta( $fp, $line, "^Subfamily:", ",\\s+\\d+\\s+bases,\\s+\\w+\\s+checksum\\.", "" );
		} else {
			return fileIterFasta( $fp, $line );
		}
	} elsif ( $line =~ /^CLUSTAL/ )
	{
		return fileIterClustal( $fp, $line );
	} elsif ( $line =~ /STOCKHOLM/ )
	{
		return fileIterStockholm( $fp, $line );
	} elsif ( $line =~ /^\S+\s+\S+\s*$/ )
	{
		# loosely matches tigrfam's SEED format
		return fileIterSeed( $fp, $line );
	}

	return undef;
}

# This is file pointer iterator specific to the stockholm format
# only the accession and sequences are saved but it can be easily
# modified to support all fields
sub fileIterStockholm
{
	my $fp = shift;
	my $line = shift;
	my %se = ();

	$line = <$fp>
		if ( !defined($line) );

	while ( defined( $line ) )
	{
		chomp( $line );
		if ( $line =~ /^#=GF\s+AC\s+(\S+)/i )
		{
			$se{acc} = $1;
			$se{seqs} = [];
		} elsif ( $line =~ /^#=GF\s+AM\s+(\S+)/i )
		{
			$se{am} = $1;
		} elsif ( $line =~ /^\/\// )
		{
			last;
		} elsif ( $line !~ /^#/ )
		{
			my ( $seqName, $seq ) = split(/\s+/, $line, 2);
			$seq =~ tr/./-/;
			my %e = ( 'name' => $seqName, 'seq' => $seq );
			push( @{$se{seqs}}, \%e );
		}

		$line = <$fp>
			if ( $line !~ /^\/\// );
	}

	return undef
		if ( !exists( $se{acc} ) );

	$se{numSeqs} = scalar( @{$se{seqs}} );
	return \%se;
}

# a fasta iterator
sub fileIterFasta
{
	my $fp = shift;
	my $line = shift;
	my $ignorePattern = shift;
	my $searchPattern = shift;
	my $replaceString = shift;
	my $type = shift;
	my %se = ( 'seqs' => [] );

	$line = <$fp>
		if ( !defined($line) );

	while ( defined( $line ) )
	{
		$line =~ s/\s+$//g;
		if ( $line =~ /^>(.+)$/ )
		{
			my $seqName = $1;
			if ( defined( $ignorePattern ) && ( $seqName =~ /$ignorePattern/i ) )
			{
				while ( $line = <$fp> )
				{
					$line =~ s/\s+$//g;
					last if ( $line =~ /^>/ );
				}
				if ( defined($line) )
				{
					( $seqName ) = $line =~ /^>(.+)$/;
				} else {
					$seqName = undef;
				}
			}
			if ( defined($seqName) )
			{
				$seqName =~ s/$searchPattern/$replaceString/g
					if ( defined($searchPattern) && defined($replaceString) );
				my %e = ( 'name' => $seqName, 'seq' => "" );
				push( @{$se{seqs}}, \%e );
			}
		} else
		{
			$line =~ s/^\s+|\s+$//g;
			$line =~ tr/./-/;
			# this is a hack for gene3d - not sure if correct
			$line =~ tr/a-z//d
				if ( defined($type) && ($type eq 'gene3d') );
			$se{seqs}->[-1]->{seq} .= $line;
		}

		$line = <$fp>;
	}

	$se{numSeqs} = scalar( @{$se{seqs}} );
	return \%se;
}

# a tigrfam seed iterator
sub fileIterSeed
{
	my $fp = shift;
	my $line = shift;
	my %se = ( 'seqs' => [] );

	$line = <$fp>
		if ( !defined($line) );

	while ( defined( $line ) )
	{
		chomp( $line );
		if ( $line =~ /^(\S+)\s+(\S+)\s*$/ )
		{
			my %e = ( 'name' => $1, 'seq' => $2 );
			$e{seq} =~ tr/./-/;
			push( @{$se{seqs}}, \%e );
		}

		$line = <$fp>;
	}

	$se{numSeqs} = scalar( @{$se{seqs}} );
	return \%se;
}

# This is file pointer iterator specific to the clustal format
sub fileIterClustal
{
	my $fp = shift;
	my $line = shift;
	my %se = ( );
	my %nameToIdx = ( );
	my $idx = 0;

	$line = <$fp>
		if ( !defined($line) );

	while ( defined( $line ) )
	{
		chomp( $line );
		last if ( ( $line =~ /^CLUSTAL/ ) && exists( $se{seqs} ) );

		if ( ( $line !~ /^\s*$/ ) && ( $line !~ /^[\s.*:]+$/ ) && ( $line !~ /^CLUSTAL/ ) )
		{
			my ( $seqName, $pSeq ) = split( /\s+/, $line, 2 );
			if ( !defined( $pSeq ) )
			{
				$seqName = substr( $line, 0, 16 );
				$pSeq = substr( $line, 16 );
				$seqName =~ s/^\s+|\s+$//g;
			}

			$se{seqs} = []
				if ( !exists( $se{seqs} ) );
			if ( !exists( $nameToIdx{$seqName} ) )
			{
				push( @{$se{seqs}}, { 'name' => $seqName, 'seq' => $pSeq } );
				$nameToIdx{$seqName} = $idx++;
			} else {
				$se{seqs}->[$nameToIdx{$seqName}]->{seq} .= $pSeq;
			}
		}

		$line = <$fp>;
	}

	return undef
		if ( !exists( $se{seqs} ) );

	$se{numSeqs} = scalar( @{$se{seqs}} );
	return \%se;
}

sub writeBFa
{
	my $prefix = shift;
	my $se = shift;
	my $stripName = shift;
	my $seqWidth = shift;
	$stripName = 0
		if ( !defined($stripName) );
	$seqWidth = 60
		if ( !defined($seqWidth) );

	my $best = getMorganBest( $se->{seqs}, $se->{numSeqs} );
	local *OUT;

	open(OUT, ">${prefix}.b");
	my $c = 0;
	my $seqLen = length( $se->{seqs}->[0]->{seq} );

	while ( $c < $seqLen )
	{
		for ( my $i = 0; $i < $se->{numSeqs}; $i++ )
		{
			my $name = $se->{seqs}->[$i]->{name};
			( $name ) = $name =~ /^(\w+)/
				if ( $stripName > 0 );
			#$name = substr( $name, 0, 33 )
			#	if ( length( $name ) > 33 );
			my $seq;
			if ( $c + $seqWidth < $seqLen )
			{
				$seq = substr( $se->{seqs}->[$i]->{seq}, $c, $seqWidth );
			} else {
				$seq = substr( $se->{seqs}->[$i]->{seq}, $c );
			}
			print OUT sprintf( "%-30s     %s\n", $name, $seq );
		}
		print OUT "\n";
		$c += $seqWidth;
	}
	close(OUT);

	open(OUT, ">${prefix}.seq");
	my $name = $best->{name};
	( $name ) = $name =~ /^(\w+)/
		if ( $stripName > 0 );
	my $seq = $best->{seq};
	$seq =~ tr/-.//d;

	print OUT File::formatFasta( $name, $seq, 60 );
	close(OUT);
}

sub reformatDb_pfam
{
	my $opts = shift;
	my $files = shift;
	my $numFiles = shift;

	my $outDir = $opts->{o} . "/pfam";
	my $pfamAmFile = $outDir . "/pfam.am";
	local *AMOUT;

	open(AMOUT, ">$pfamAmFile");
	for ( my $i = 0; $i < $numFiles; $i++ )
	{
		my $file = $files->[$i];
		local *IN;
		File::fileGzOpen( \*IN, $file );

		#print "Processing: [$file] ...\n"
		#	if ( !exists( $opts->{q} ) );

		while ( my $se = alnFileIter( \*IN ) )
		{
			my $prefix = $outDir . "/" . $se->{acc};
			writeBFa( $prefix, $se );
			if ( exists( $se->{am} ) )
			{
				print AMOUT join("\t", $se->{acc}, $se->{am}), "\n";
			} else {
				print STDERR "Warning: PFAM accession '$se->{acc}' did not have an AM line\n";
			}
		}

		close(IN);
	}
	close(AMOUT);
}

sub reformatDb_smart
{
	my $opts = shift;
	my $files = shift;
	my $numFiles = shift;

	my $outDir = $opts->{o} . "/smart";

	for ( my $i = 0; $i < $numFiles; $i++ )
	{
		my $file = $files->[$i];
		local *IN;
		File::fileGzOpen( \*IN, $file );

		#print "Processing: [$file] ...\n"
		#	if ( !exists( $opts->{q} ) );

		my ( $baseFile ) = $file =~ /\/([^\/]+)\.aln$/;

		while ( my $se = alnFileIter( \*IN ) )
		{
			my $prefix = $outDir . "/" . $baseFile;
			writeBFa( $prefix, $se );
		}

		close(IN);
	}
}

sub reformatDb_gene3d
{
	my $opts = shift;
	my $type = shift;
	my $files = shift;
	my $numFiles = shift;

	my $outDir = $opts->{o} . "/gene3d";
	mkdir( $outDir, 0755 )
		if ( !(-d $outDir) );

	print "Processing alignment files for '$type' ...\n"
		if ( !exists( $opts->{q} ) );
	for ( my $i = 0; $i < $numFiles; $i++ )
	{
		my $file = $files->[$i];
		local *IN;
		File::fileGzOpen( \*IN, $file );

		#print "Processing: [$file] ...\n"
		#	if ( !exists( $opts->{q} ) );

		my ( $baseFile ) = $file =~ /\/([^\/]+)\.alignment$/;

		while ( my $se = alnFileIter( \*IN, $type ) )
		{
			my $prefix = $outDir . "/" . $baseFile;
			writeBFa( $prefix, $se );
		}

		close(IN);
	}
}

sub reformatDb_panther
{
	my $opts = shift;
	my $files = shift;
	my $numFiles = shift;

	my $outDir = $opts->{o} . "/panther";

	for ( my $i = 0; $i < $numFiles; $i++ )
	{
		my $file = $files->[$i];
		local *IN;
		File::fileGzOpen( \*IN, $file );

		#print "Processing: [$file] ...\n"
		#	if ( !exists( $opts->{q} ) );

		my ( $baseFile ) = $file =~ /\/?([^\/]+)\/cluster\.pir$/;

		while ( my $se = alnFileIter( \*IN ) )
		{
			my $prefix = $outDir . "/" . $baseFile;
			writeBFa( $prefix, $se );
		}

		close(IN);
	}
}

# slow since we have to run muscle for each seed file =(
sub reformatDb_pirsf
{
	my $opts = shift;
	my $files = shift;
	my $numFiles = shift;

	my $outDir = $opts->{o} . "/pirsf";

	for ( my $i = 0; $i < $numFiles; $i++ )
	{
		my $file = $files->[$i];
		local *IN;
		open(IN, "$ENV{FASTHMM_DIR}/bin/muscle -quiet -in $file |");

		#print "Processing: [$file] ...\n"
		#	if ( !exists( $opts->{q} ) );

		my ( $baseFile ) = $file =~ /\/([^\/]+)\.seed$/;

		while ( my $se = alnFileIter( \*IN ) )
		{
			my $prefix = $outDir . "/" . $baseFile;
			# 1 parameter tells writeBFa to strip the defline
			# to the first \w+
			writeBFa( $prefix, $se, 1 );
		}

		close(IN);
	}
}

sub reformatDb_tigrfam
{
	my $opts = shift;
	my $files = shift;
	my $numFiles = shift;

	my $outDir = $opts->{o} . "/tigrfam";

	for ( my $i = 0; $i < $numFiles; $i++ )
	{
		my $file = $files->[$i];
		local *IN;
		File::fileGzOpen( \*IN, $file );

		#print "Processing: [$file] ...\n"
		#	if ( !exists( $opts->{q} ) );

		my ( $baseFile ) = $file =~ /\/([^\/]+)\.SEED$/;

		while ( my $se = alnFileIter( \*IN ) )
		{
			my $prefix = $outDir . "/" . $baseFile;
			writeBFa( $prefix, $se );
		}

		close(IN);
	}
}

sub reformatDb
{
	my $opts = shift;
	my $type = shift;
	my $files = shift;
	my $numFiles = shift;

	#gene3d|panther|pfam|pirsf|smart|tigrfam
	if ( $type eq 'gene3d' )
	{
		reformatDb_gene3d( $opts, $type, $files, $numFiles );
	} elsif ( $type eq 'panther' )
	{
		reformatDb_panther( $opts, $files, $numFiles );
	} elsif ( $type eq 'pfam' )
	{
		reformatDb_pfam( $opts, $files, $numFiles );
	} elsif ( $type eq 'pirsf' )
	{
		reformatDb_pirsf( $opts, $files, $numFiles );
	} elsif ( $type eq 'smart' )
	{
		reformatDb_smart( $opts, $files, $numFiles );
	} elsif ( $type eq 'tigrfam' )
	{
		reformatDb_tigrfam( $opts, $files, $numFiles );
	} else {
		print STDERR "Unknown type '", $type, "' - aborting\n";
	}
}
