#
#  $Id: Args.pm,v 1.2 2007/10/16 22:48:09 whuang Exp $
#  fastHmm/fastBlast Alignment Tools
#  http://microbesonline.org/fasthmm (fasthmm@microbesonline.org)
#
#  Argument parser module
#
#  Copyright (C) 2001-2007 Samuel L. Rash
#  All rights reserved.
#
#  This library is free software; you can redistribute it and/or
#  modify it under the terms of the GNU Lesser General Public
#  License as published by the Free Software Foundation; either
#  version 2.1 of the License, or (at your option) any later version.
#
#  This library is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  Lesser General Public License for more details.
#
#  You should have received a copy of the GNU Lesser General Public
#  License along with this library; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
#  USA.
#
#  Disclaimer
#
#  NEITHER THE UNITED STATES NOR THE UNITED STATES DEPARTMENT OF ENERGY,
#  NOR ANY OF THEIR EMPLOYEES, MAKES ANY WARRANTY, EXPRESS OR IMPLIED,
#  OR ASSUMES ANY LEGAL LIABILITY OR RESPONSIBILITY FOR THE ACCURACY,
#  COMPLETENESS, OR USEFULNESS OF ANY INFORMATION, APPARATUS, PRODUCT,
#  OR PROCESS DISCLOSED, OR REPRESENTS THAT ITS USE WOULD NOT INFRINGE
#  PRIVATELY OWNED RIGHTS.
#

package Args;
require Exporter;

use strict;

use vars '$VERSION';
$VERSION = 0.01;

our @ISA = qw(Exporter);
our @EXPORT = qw();


#
# Function prototypes
#
sub getArgs($\@;$$$$$); #($pStr, @ARGV, $reqOptStr, $usageStr, $errorFatal, 
			#			$dispVal, $verbose)
sub procReqOptStr($\%); #($reqOptStr, %opts)
sub procArgStr($\%); 	#($pStr, %optInfo)
sub depStrToExpr($); 	#($depStr)
sub checkOpts(\%\%$); 	#(%opts, %optInfo, $usageStr)
sub dispParams(\%\@); 	#(%opts)
sub dispOptInfo($); 	#($pStr)
sub dispOptInfoHash(\%);#(%optInfo)

#
# Returns an array with ref to hash and ref to array. 1st element is ref to
# hash that has each dash-option as a key and then its value.  the ref to
# array is a list of non-dash options;  
#
#	$pStr is "[+[;depStr;]]<str>[:[<type>]][;<depStr]|..."
#	+ indicates a required option.  If specified w/o a depStr, it is an
#	unconditional requirement.  Otherwise, the option is required only if
#	the depStr evals to true (see below for definition of depStr).  <str>
#	is the option you want w/o the dash (ex: x for -x).  If a : is
#	present, it means a value must be specified.  If <type>	follows the
#	value, it may be either "p", "i", "n", "w", "o" which means the arg
#	MUST be	a POSITIVE integer, an integer, number (int or float), or
#	word only (a-zA-z0-9_-), or optional value (of any type) respectively.
#	No <type> means it can be anything (type "a", also allowed).  If a
#	; <depStr> appears following <str>, this is an expression on the
#	other options that must eval to true.  Other options are referred to
#	simply by their <str>.  The expression may be a logical expression
#	using *, +, ! to mean and, or, not, respectively.  It may also include
#	parens ().  The expression is evaluated as being an expression on 
#	whether or not other options are defined.
#
#	Example:
#
#	"+;!n;p:w|d:w;i*o|i;d*o|o;d*i|n"
#
#	means -p is required if -n is not used, and must be only word chars.
#	-d is word only and if it is defined, -i and -o must be also.  -i and
#	-o have no type but must be defined if either other one is or -d is.
#
#	$reqOptStr is a string of the form <defNum>|<num>;<depStr>|...
#	ie, pipe separated "clauses".  At most one clause with only a num
#	should appear as this is the default number of required opts.  If more
#	than one appears, the last will be used as default.  Then, each clause
#	is eval'd.  The first one found to be true is used (the <num>).  Else,
#	the default value is used.  No default means a default of 0.
#
#	Access to optional values via %opts:  If a value is boolean, it will
#	be undef/1.  If it is a valued type, i,n,w,p then $opts->{key} will be
#	the value.  If type is "o", and the option is present, $opts->{key}
#	will be undef, but true when tested with exists().  If a value is
#	present, it will be the same as i,n,w,p.
#
sub getArgs($\@;$$$$$)	#($pStr, @ARGV, $reqOpt, $usageStr, $errorFatal, 
			#				$dispVal, $verbose)
{
	my ($pStr, $argv, $reqOptStr, $usageStr,
		$errorFatal, $dispVal, $verbose) = @_;
	$errorFatal = 1
		unless defined($errorFatal);
	$reqOptStr = -1 #no required args unless specified
		unless defined($reqOptStr);

	#first, remove whitespace from the string
	$pStr =~ s/\s+//g;

	my @nonOpts = ();
	my %opts = ();

	# If we are in verbose mode, concatenate info about option string to
	# usage
	$usageStr .= "option Info:\n" . dispOptInfo($pStr)
		if ( defined($verbose) && ($verbose == 1) );

	# Build up info about each option into a hash from arg string
	my %optInfo;
	procArgStr($pStr, %optInfo);

	# Scan args and put in approriate hash
	for ( my $i = 0; $i < @$argv; $i++ )
	{
		# if this is a dash-option
		if ( $argv->[$i] =~ /^-[a-zA-z][a-zA-Z0-9]*$/ )
		{
			my $opt = $argv->[$i];
			# Get actual option by removing the dash
			$opt =~ s/^-//;

			# Check if this option takes a value
			if ( defined( $optInfo{$opt} ) &&
				( $optInfo{$opt}->{type} ne "b" ) )
			{
				# Check if optional value
				if ( $optInfo{$opt}->{type} eq "o" )
				{
					if ( ( $i < $#{@$argv} ) &&
						!($argv->[$i+1] =~ /^-/) )
					{
						# Optionally takes a value 
						$opts{$opt} = ${@$argv}[$i+1];
						$i++;
					} else {
						$opts{$opt} = undef;
					}
				} else {
					# Requires an argument
					die "$usageStr\noption [${@$argv}[$i]] requires a value of type " . 
						"[$optInfo{$opt}{type}] afterwards\n"
						if $i == $#{@$argv};
					$opts{$opt} = ${@$argv}[$i+1];
					$i++;
				}
			} elsif ( defined($optInfo{$opt}) )
			{
				# This is a boolean option (exists or not)
				$opts{$opt} = 1;
			} else
			{
				# Invalid option
				die "$usageStr\nunrecognized option: [${@$argv}[$i]]\n";
			}
		} else {
			push( @nonOpts, $argv->[$i] );
		}
	}

	# Figure out if we are doing any required option checking
	# if $numReqOpt == -1, we aren't requiring any options
	# (ie, do no checking)
	my ( $cond, $numReqOpt ) = procReqOptStr( $reqOptStr, %opts );

	# Check options first
	if( checkOpts( %opts, %optInfo, $usageStr ) == 0 )
	{
		exit(1) 
			if $errorFatal;

		# Return undef to indicate failure
		return undef;
	}

	# Check nonOption count
	my $numOptGiven = scalar(@nonOpts);
	if ( ( $numReqOpt != -1 ) &&
		!eval("$numOptGiven $cond $numReqOpt") )
	{
		print $usageStr;
		if ( scalar( @nonOpts ) > $numReqOpt )
		{
			print "too many arguments [need: $numReqOpt, got: ", scalar(@nonOpts), "]\n";
		} else {
			print "not enough arguments [need: $cond $numReqOpt]\n";
		}

		exit(1)
			if($errorFatal);

		# Return undef on error
		return undef;
	}

	dispParams( %opts, @nonOpts )
		if ( defined( $dispVal ) && ( $dispVal == 1 ) );

	# All checks okay, return values
	return ( \%opts, \@nonOpts );
}

# Processes a required option string to determine how many required options
# there are given which options are defined
sub procReqOptStr($\%)	#($reqOptStr, %opts)
{
	my ($reqOptStr, $opts) = @_;
	my $retNum = -1; #ie, no required number of args
	my $retCond = "==";

	foreach my $clause ( split(/\|/, $reqOptStr) )
	{
		my ($num, $depStr) = split( /\;/, $clause );
		my $cond;

		if ( $num =~ /^\s*(>=|<=|>|<)\s*(\d+)\s*$/ )
		{
			$num = $2;
			$cond = $1;
		} else {
			$cond = "==";
		}

		if ( defined($depStr) )
		{
			if ( eval( depStrToExpr( $depStr ) ) )
			{
				$retNum = $num;
				$retCond = $cond;
				last;
			}
		} else {
			$retNum = $num;
			$retCond = $cond;
		}
	}

	return ( $retCond, $retNum );
}

# Processes an argument string and builds up 
# %optInfo - hash with valid options (w/o dash) as keys.
sub procArgStr($\%)	#($pStr, %optInfo)
{
	my ( $pStr, $optInfo ) = @_;

	foreach my $item ( split(/\|/, $pStr) )
	{
		# First, is this option required?
		my $req = 0;
		if ( $item =~ /^\+/ )
		{
			my ( $reqStr ) = ( $item =~ /^\+\;(.*?)\;/ );

			# Check if an expression has been passed in
			if ( defined($reqStr) )
			{
				# Store string for later evaluation
				$req = $reqStr;
			} else {
				$req = 1;
			}

			$item =~ s/^\+;.*?;/+/;
		}

		$item =~ s/^\+//;

		# Now parse out info
		my ( $opt, $type, $depStr ) = 
			( $item =~ /(\w+)(\:[inwao]?)?(?:\;(.*))?/ );

		$type = "b" #boolean--no value
			unless defined( $type );

		# Type may be empty which means "a"
		$type = ":a"
			if ( $type eq ":" );

		$type =~ s/^://; # Take of leading colon
		$type = lc($type);

		# Now process $depStr--change all *,+ to && and || and
		# replaced option with defined(opt)
		$depStr = ""
			unless defined($depStr);
		if( !defined( $optInfo->{$opt} ) )
		{
			# Store hash with info about this option
			# (required?  type?  dependency string?)
			$optInfo->{$opt} = {
				'req' => $req,
				'type' => $type,
				'depStr' => $depStr
					   };
		} else {
			print STDERR "ERROR: getArgs(): option [$opt] redefined\n";
		}
	}
}

# Takes a boolean expression on a set of keys in a hashref $opts and changes
# From x*y+z -> defined($opts->{"x"}) && defined($opts->{"y"}) ||
# defined($opts->{"z"})
sub depStrToExpr($)	#($depStr)
{
	my ( $depStr ) = @_;

	# First replace *,+ with &&,||
	$depStr =~ s/\*/ \&\& /g;
	$depStr =~ s/\+/ \|\| /g;

	# We assume a hash reference $opts exists when this string is eval'd
	$depStr =~ s/([a-zA-z]+)/defined\(\$opts->\{"$1"\}\)/g;		
	return $depStr;
}

# Given a processed arg hash %optInfo (data struct defining opts/etc) and
# actual passed in options + values, checks for validity
sub checkOpts(\%\%$)	#(%opts, %optInfo, $usageStr)
{
	my ( $opts, $optInfo, $usageStr ) = @_;

	foreach my $opt ( keys( %{$optInfo} ) )
	{
		# Check if required and if so, if defined
		if ( eval( depStrToExpr( $optInfo->{$opt}->{req} ) ) &&
			!exists( $opts->{$opt} ) )
		{
			print $usageStr;
			print "WARNING: given specified options/arguments,\noption [$opt] is required and not specified\n";

			return 0;
		} elsif ( defined( $opts->{$opt} ) )
		{
			my $error = 0;

			# First check type
			if ( ($optInfo->{$opt}->{type} eq "i") &&
				!( $opts->{$opt} =~ /^\-?\d+$/ ) )
			{
				print $usageStr;

				# Required type is int and actual
				# type doesn't match
				print "option [$opt] must be of type integer\n";
				$error = 1;
			} elsif ( ($optInfo->{$opt}->{type} eq "p") &&
				!( $opts->{$opt} =~ /^\d+$/ ) )
			{
				print $usageStr;

				# Required type is int and actual
				# type does not match
				print "option [$opt] must be of type integer\n";
				$error = 1;
			} elsif ( ($optInfo->{$opt}->{type} eq "n") &&
				!( $opts->{$opt} =~ /(^\d+\.?\d*$)|(^\d*.?\d+$)/ ) )
			{
				print $usageStr;

				# Required type is number and actual
				# type does not match
				print "option [$opt] must be of type number (int or float)\n";
				$error = 1;
			} elsif ( ( $optInfo->{$opt}->{type} eq "w") &&
				!( $opts->{$opt} =~ /^[a-zA-Z_\-]+$/ ) )
			{
				print $usageStr;

				# Required type is word and actual
				# type does not match
				print "option [$opt] must be of type word [a-zA-z_-]\n";
				$error = 1;
			}

			# Now check dependencies
			if ( ( $optInfo->{$opt}{depStr} ne "" ) &&
				!eval( depStrToExpr( $optInfo->{$opt}->{depStr} ) ) )
			{
				print $usageStr;
				print "option [$opt] fails dependencies\n";
				my %h = ($opt => $optInfo->{$opt});
				print dispOptInfoHash(%h);
				$error = 1;
			}

			return 0
				if $error;
		}
	}

	# If we make it here, everything is okay
	return 1;
}

# Prints out option values
sub dispParams(\%\@)	#(%opts)
{
	my ( $opts, $nonOpts ) = @_;
	print "-" x 79 . "\n";
	print "options specified:\n";
	foreach my $opt ( keys( %{$opts} ) )
	{
		print "$opt -> [$opts->{$opt}]\n";
	}
	print "\n" . "-" x 79 . "\n";
	print "non-options:\n";
	print join("\n", @{$nonOpts}), "\n";
	print "-" x 79 . "\n\n";
}

# Given an option arg string, print out english readable version of it
sub dispOptInfo($)	#($pStr)
{
	my ( $pStr ) = @_;
	my %optInfo;
	procArgStr($pStr, %optInfo);
	return( dispOptInfoHash( %optInfo ) );
}

# Display english readable verison of option info hash
sub dispOptInfoHash(\%)	#(%optInfo)
{
	my ( $optInfo ) = @_;
	my $retStr = "";

	foreach my $opt ( keys( %{$optInfo} ) )
	{
		$retStr .= "option [$opt]\n" . 
				"\trequired: [$optInfo->{$opt}{req}]\n" .
				"\ttype:     [$optInfo->{$opt}{type}]\n" .
				"\tdepStr:   [$optInfo->{$opt}{depStr}]\n";
	
	}

	return $retStr;
}

1;
