#
#  $Id: Cache.pm,v 1.3 2007/10/16 22:24:22 whuang Exp $
#  fastHmm/fastBlast Alignment Tools
#  http://microbesonline.org/fasthmm (fasthmm@microbesonline.org)
#
#  Sequential file content caching and iterator module
#
#  Copyright (C) 2007 The Regents of the University of California
#  All rights reserved.
#
#  This library is free software; you can redistribute it and/or
#  modify it under the terms of the GNU Lesser General Public
#  License as published by the Free Software Foundation; either
#  version 2.1 of the License, or (at your option) any later version.
#
#  This library is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  Lesser General Public License for more details.
#
#  You should have received a copy of the GNU Lesser General Public
#  License along with this library; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
#  USA.
#
#  Disclaimer
#
#  NEITHER THE UNITED STATES NOR THE UNITED STATES DEPARTMENT OF ENERGY,
#  NOR ANY OF THEIR EMPLOYEES, MAKES ANY WARRANTY, EXPRESS OR IMPLIED,
#  OR ASSUMES ANY LEGAL LIABILITY OR RESPONSIBILITY FOR THE ACCURACY,
#  COMPLETENESS, OR USEFULNESS OF ANY INFORMATION, APPARATUS, PRODUCT,
#  OR PROCESS DISCLOSED, OR REPRESENTS THAT ITS USE WOULD NOT INFRINGE
#  PRIVATELY OWNED RIGHTS.
#

package Cache;
require Exporter;

use strict;

use vars '$VERSION';
$VERSION = 0.01;

our @ISA = qw(Exporter);
our @EXPORT = qw();

our @caches = ();
our $nextCacheId = 0;

sub initCache
{
	my $maxElements = shift;

	my $id = $nextCacheId++;
	$caches[$id] = {};
	$caches[$id]->{max} = $maxElements;
	$caches[$id]->{hits} = 0;
	$caches[$id]->{requests} = 0;
	$caches[$id]->{elements} = {};
	$caches[$id]->{usage} = {};

	return $id;
}

sub addCache
{
	my $id = shift;
	my $elemId = shift;
	my $elemData = shift;

	$caches[$id]->{requests}++;
	if ( existsCache( $id, $elemId ) )
	{
		# replace the data and reset usage
		$caches[$id]->{usage}->{$elemId} = 1;
		$caches[$id]->{elements}->{$elemId} = $elemData;
	} else {
		my $numCacheElems = scalar( keys( %{$caches[$id]->{usage}} ) );
		if ( $numCacheElems < $caches[$id]->{max} )
		{
			$caches[$id]->{usage}->{$elemId} = 1;
			$caches[$id]->{elements}->{$elemId} = $elemData;
		} else {
			# LRU
			my $minKey = ( keys( %{$caches[$id]->{usage}} ) )[0];
			my $min = $caches[$id]->{usage}->{$minKey};
			foreach my $_elemId ( keys( %{$caches[$id]->{usage}} ) )
			{
				if ( $caches[$id]->{usage}->{$_elemId} < $min )
				{
					$min = $caches[$id]->{usage}->{$_elemId};
					$minKey = $_elemId;
				}
			}

			delete( $caches[$id]->{usage}->{$minKey} );
			delete( $caches[$id]->{elements}->{$minKey} );

			$caches[$id]->{usage}->{$elemId} = 1;
			$caches[$id]->{elements}->{$elemId} = $elemData;
		}
	}
}

sub getCache
{
	my $id = shift;
	my $elemId = shift;

	if ( defined($id) && defined($elemId) && ($id < $nextCacheId) && exists( $caches[$id]->{elements}->{$elemId} ) )
	{
		$caches[$id]->{usage}->{$elemId}++;
		$caches[$id]->{hits}++;
		$caches[$id]->{requests}++;
		return $caches[$id]->{elements}->{$elemId};
	}

	return undef;
}

sub existsCache
{
	my $id = shift;
	my $elemId = shift;

	if ( defined($id) && defined($elemId) && ($id < $nextCacheId) )
	{
		return exists( $caches[$id]->{elements}->{$elemId} );
	}

	return 0;
}

sub statsCache
{
	my $id = shift;

	if ( defined($id) && ($id < $nextCacheId) )
	{
		print STDERR sprintf( "reqs: %d, hits: %d, hit %%: %.2f\n", $caches[$id]->{requests}, $caches[$id]->{hits}, (100.0 * $caches[$id]->{hits} / $caches[$id]->{requests}) );
	}
}

sub flushCache
{
	my $id = shift;

	if ( defined($id) && ($id < $nextCacheId) )
	{
		delete( $caches[$id]->{elements} );
		delete( $caches[$id]->{usage} );
		$caches[$id]->{elements} = {};
		$caches[$id]->{usage} = {};
		$caches[$id]->{hits} = 0;
		$caches[$id]->{requests} = 0;
	}
}



1;
