#!/usr/bin/perl -w

use strict;
use Args;

#
# USAGE
#
my $usage =
"Usage:
  install.pl <options>

Parameters:
  -prefix  <path>	Specify base directory in which to install fastHmm

Optional Parameters:
  -out     <path>       New directory to create fasthmm installation in
                          (Default: prefix/fasthmm)
  -perl    <perlexe>	Specify full path to Perl interpreter
			  (Default: /usr/bin/perl)
  -ncbi    <path>	Specify bin directory for NCBI blast executables
  -cdhit   <path>	Specify bin directory for cd-hit executables
  -hmmer   <path>	Specify bin directory for hmmer executables
  -muscle  <path>	Specify bin directory for MUSCLE executable
  -sys     <path1:...>	Specify paths to search for UNIX-style executables
			  (cat, cp, cut, gzip, md5sum, mv, rm, sort, tar)

  * By default, if some or all of the optional parameters are not specified,
    the installer will automatically search your existing \$PATH.  If one
    or more of the required tools are not located, the installer will not be
    able to complete.

";


#
# Globals/Constants
#

# List of tools for each "class" or group of tools
my %tools = (
		'ncbi'	=> [ 'blastall', 'blastpgp', 'fastacmd', 'formatdb', 'rpsblast'],
		'cdhit'	=> [ 'cd-hit', 'clstr_rev.pl', 'mcd-hit', ],
		'hmmer'	=> [ 'hmmsearch', ],
		'muscle'=> [ 'muscle', ],
		'sys'	=> [ 'cat', 'cp', 'cut', 'gzip', 'md5sum', 'mv', 'rm', 'sort', 'tar' ],
	    );

# Options to use to test validity of each program
# Output of tool must contain at least one line which matches 'expect'
my %expect = (
		'gzip'		=> { 'opt' => '--help', 'expect' => 'gnu\.org', },
		'rm'		=> { 'opt' => '--help', 'expect' => 'gnu\.org', },
		'sort'		=> { 'opt' => '--help', 'expect' => 'gnu\.org', },
		'tar'		=> { 'opt' => '--help', 'expect' => 'gnu\.org', },
		'md5sum'	=> { 'opt' => '--help', 'expect' => 'gnu\.org', },
		'blastall'	=> { 'opt' => '--help', 'expect' => '^blastall\s+\d+\.\d+\.\w+\s+arguments', },
		'blastpgp'	=> { 'opt' => '--help', 'expect' => '^blastpgp\s+\d+\.\d+\.\w+\s+arguments', },
		'fastacmd'	=> { 'opt' => '--help', 'expect' => '^fastacmd\s+\d+\.\d+\.\w+\s+arguments', },
		'formatdb'	=> { 'opt' => '--help', 'expect' => '^formatdb\s+\d+\.\d+\.\w+\s+arguments', },
		'cd-hit'	=> { 'opt' => '-h', 'expect' => 'Weizhong', },
		'clstr_rev.pl'	=> { 'opt' => '2>&1', 'expect' => 'Can\s*not\s+open\s+file', },
		'mcd-hit'	=> { 'opt' => '-h', 'expect' => 'Weizhong', },
		'hmmsearch'	=> { 'opt' => '-h', 'expect' => '^HMMER\s+\d+\.\d+\.\w+', },
		'muscle'	=> { 'opt' => '-version', 'expect' => '^MUSCLE\s+v\d+\.\w+', },
	     );

#
# Parse Command Line Options
#
my ($opts, $nonOpts) = Args::getArgs( 
	"+prefix:|ncbi:|cdhit:|hmmer:|muscle:|sys:|perl:|out:",
	@ARGV, -1, $usage );

die "Error: Specified -prefix does not exist '$opts->{prefix}'"
	if ( !(-d $opts->{prefix}) );

die "Error: Cannot write to -prefix '$opts->{prefix}' - check permissions"
	if ( !(-w $opts->{prefix}) );

die "Error: The installer must be run from the directory containing the fastHmm archive"
	if ( !(-x "./install.pl") || !(-r "./fastHmm_package.tar.gz") ||
		!(-r "./MD5SUM") );

foreach my $param ( qw/ncbi cdhit hmmer muscle sys/ )
{
	$opts->{$param} = exists( $ENV{PATH} ) ?
				$ENV{PATH} :
				"./"
		if ( !exists( $opts->{$param} ) );
}

# default perl is /usr/bin/perl
$opts->{perl} = "/usr/bin/perl"
	if ( !exists( $opts->{perl} ) );

install( $opts, \%tools, \%expect );

exit(0);

sub verifyChecksum
{
	my $archive = shift;
	my $sumFile = shift;
	my $md5sum = shift;

	local *IN;
	open( IN, "<$sumFile" );
	my $line = <IN>;
	close( IN );
	my ( $refSum ) = $line =~ /^(\w+)/;

	open( IN, "$md5sum -b $archive |" );
	$line = <IN>;
	close( IN );
	my ( $sum ) = $line =~ /^(\w+)/;

	return ( $sum eq $refSum );
}

sub install
{
	my $opts = shift;
	my $tools = shift;
	my $expect = shift;

	# Call does not return unless all tools resolved and tested okay
	my $rTools = resolveToolPath( $opts, $tools, $expect );

	# Check checksum of archive
	my $archive = "./fastHmm_package.tar.gz";
	my $sumFile = "./MD5SUM";
	die "Error: Archive checksum mismatch; unable to proceed!"
		if ( !verifyChecksum( $archive, $sumFile, $rTools->{md5sum} ) );

	# Everything appears to be okay; begin installation
	my $fastHmmDir = $opts->{prefix} . "/fasthmm";
	$fastHmmDir = $opts->{out} if exists $opts->{out};
	mkdir( $fastHmmDir, 0755 );
        die "Cannot create directory $fastHmmDir" unless -d $fastHmmDir;

	# Extract main archive
	print "Extracting fastHmm archive ...\n";
	my $cmd = $rTools->{tar} . " zxf $archive -C $fastHmmDir/";
	system( $cmd );

	# modify perl interpreter in perl scripts
	my @scripts = <$fastHmmDir/bin/*.pl>;
	print "Updating perl interpreter references in scripts ...\n";
	foreach my $script ( @scripts )
	{
		rename( $script, $script . ".orig" );
		local *IN;
		local *OUT;
		open(IN, "<${script}.orig");
		open(OUT, ">$script");

		print OUT "#!" . $opts->{perl} . " -w\n";
		while(<IN>)
		{
			next if ( /^#!/ );

			print OUT;
		}
		close(IN);
		close(OUT);

		unlink( $script . ".orig" );
		chmod( 0755, $script );
	}

	# Add bin symlinks
	print "Creating tool symlinks ...\n";
	foreach my $tool ( keys( %{$rTools} ) )
	{
		symlink( $rTools->{$tool}, "$fastHmmDir/bin/$tool" );
	}

	print <<END
Installed FastHMM/FastBLAST into $fastHmmDir
To compile FastHMM/FastBLAST executables please do this:
	cd $fastHmmDir; make

If you have any problems, you may need to install a C compiler (such
as gcc) or tweak the Makefile to use your compiler instead of cc.

To run FastHMM you will also need to install FastHMM databases
END
    ;
}

sub expectedOutput
{
	my $bin = shift;
	my $binExpect = shift;

	die "No expected value for $bin" unless defined $binExpect->{opt};
	my $cmd = $bin . " " . $binExpect->{opt};
	my $pattern = $binExpect->{expect};
	my $pass = 0;
	local *PIPE;

	open( PIPE, "$cmd |" );
	while( <PIPE> )
	{
		$pass++
			if ( /$pattern/ );
	}
	close( PIPE );

	return ( $pass > 0 );
}

sub resolveToolPath
{
	my $opts = shift;
	my $tools = shift;
	my $expect = shift;

	my %tools = ();

	foreach my $toolClass ( keys( %{$tools} ) )
	{
		my @paths = split(/\s*:\s*/, $opts->{$toolClass});
		foreach my $tool ( @{$tools->{$toolClass}} )
		{
			# find tool
			my $found = 0;
			foreach my $path ( @paths )
			{
				$path .= "/"
					if ( $path !~ /\/$/ );
				my $toolCheck = $path . $tool;
				if ( (-x $toolCheck)
				     && (!exists $expect{$tool} || expectedOutput( $toolCheck, $expect{$tool} )))
				{
				    $found = 1;
				    $tools{$tool} = $toolCheck;
				    last;
				}
			}

			die "Error: Could not find working copy of '$tool' using paths:\n       " . $opts->{$toolClass} . "\n"
				if ( !$found );
		}
	}

	return \%tools;
}
